﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;
using UnityEngine;

//AudioManager is a singleton class responsible for handling all game audio
public class AudioManager : MonoBehaviour
{
    public Dictionary<string, AudioFile> LoadedFiles = new Dictionary<string, AudioFile>();
    public Dictionary<string, AudioGroup> AudioGroups = new Dictionary<string, AudioGroup>();

    private AudioSource _MusicSource;   //The source of music
    private List<AudioSource> _SFXSources = new List<AudioSource>();    //A list of sources to play multiple SFXs at one time
    private List<AudioSource> _PausedSFXSources = new List<AudioSource>();
    private bool _DetectedMusicStopped = true;

    //Singleton
    private static AudioManager _Instance;
    public static AudioManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<AudioManager>();
            }

            return _Instance;
        }
    }

    #region Events

    #region Music Finished
    public class MusicFinishedEventArgs : EventArgs
    {
        public AudioClip Clip { get; private set; }

        public MusicFinishedEventArgs(AudioClip clip)
        {
            Clip = clip;
        }
    }

    private void OnMusicFinished(MusicFinishedEventArgs e)
    {
        EventHandler<MusicFinishedEventArgs> handler = MusicFinished;
        if (handler != null)
        {
            handler(this, e);
        }
    }

    public event EventHandler<MusicFinishedEventArgs> MusicFinished;
    #endregion

    #endregion

    /// <summary>
    /// Initializes the AudioManager by setting up sources and parsing audio files
    /// </summary>
    /// <returns>Was the initialization successful?</returns>
    public bool Initialize()
    {
        try
        {
            //Setup the music source
            _MusicSource = Instance.gameObject.AddComponent<AudioSource>();
            _MusicSource.loop = true;
            _MusicSource.volume = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.MusicVolume]);
            return ParseAudioFiles();
        }

        catch (Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }

    /// <summary>
    /// Plays the audio clip as a music file, on loop and at the set music volume
    /// </summary>
    /// <param name="clip">The clip to play as music</param>
    public void PlayMusic(AudioClip clip, bool shouldLoop = true)
    {
        _DetectedMusicStopped = false;
        _MusicSource.clip = clip;
        _MusicSource.loop = shouldLoop;
        _MusicSource.Play();
    }

    /// <summary>
    /// Stops the music source playing
    /// </summary>
    public void StopMusic()
    {
        _MusicSource.Stop();
    }

    /// <summary>
    /// Pauses the music source
    /// </summary>
    public void PauseMusic()
    {
        _MusicSource.Pause();
    }

    /// <summary>
    /// Unpauses the music source
    /// </summary>
    public void UnPauseMusic()
    {
        _MusicSource.UnPause();
    }

    /// <summary>
    /// Plays an SFX clip
    /// </summary>
    /// <param name="clip">The clip to play</param>
    /// <param name="shouldLoop">Should the clip loop?</param>
    /// <param name="volume">How loud should the clip be?</param>
    /// <returns>An audio source for the clip</returns>
    public AudioSource PlaySFX(AudioClip clip, bool shouldLoop = false, float volume = -1.0f)
    {
        AudioSource source = _SFXSources.FirstOrDefault(x => !x.isPlaying); //Get the first SFX source that isn't playing

        if (source == null || source == default(AudioSource))
        {
            //Couldn't find an SFX source that isn't playing, let's make a new one
            source = Instance.gameObject.AddComponent<AudioSource>();
            _SFXSources.Add(source);
        }

        source.Stop();

        if (volume < 0.0f || volume > 1.0f)
        {
            source.volume = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.SFXVolume]);  //Set the volume of the source to be the SFX volume setting
        }

        else
        {
            source.volume = volume;
        }

        if (shouldLoop)
        {
            //Loop, so set the clip and play
            source.loop = true;
            source.clip = clip;
            source.Play();
        }

        else
        {
            //Not loop, so just PlayOneShot
            source.PlayOneShot(clip);
        }

        return source;
    }

    /// <summary>
    /// Pauses an SFX using its source
    /// </summary>
    /// <param name="sourceToPause">The SFX source to pause</param>
    public void PauseSFX(AudioSource sourceToPause)
    {
        foreach(AudioSource sfxSource in _SFXSources.ToList())
        {
            if(sourceToPause == sfxSource && !IsSFXPaused(sfxSource))
            {
                //Found it, let's pause it and shuffle the source around in the lists
                sfxSource.Pause();
                _SFXSources.Remove(sfxSource);
                _PausedSFXSources.Add(sfxSource);
                break;
            }
        }
    }

    /// <summary>
    /// Returns whether the specified SFX source is paused
    /// </summary>
    /// <param name="source">The source to check</param>
    /// <returns>Is the SFX source paused?</returns>
    public bool IsSFXPaused(AudioSource source)
    {
        foreach(AudioSource pausedSource in _PausedSFXSources)
        {
            if(pausedSource.clip == source.clip)
            {
                //Found it, it's paused
                return true;
            }
        }

        return false;   //Didn't find it, it's not paused
    }

    /// <summary>
    /// Unpauses an SFX source
    /// </summary>
    /// <param name="sourceToResume">The SFX source to resume</param>
    public void UnPauseSFX(AudioSource sourceToResume)
    {
        foreach (AudioSource sfxSource in _PausedSFXSources.ToList())
        {
            if (sourceToResume == sfxSource)
            {
                //Found it, let's unpause it and shuffle the source around in the lists
                sfxSource.UnPause();
                _PausedSFXSources.Remove(sourceToResume);
                _SFXSources.Add(sourceToResume);
                break;
            }
        }
    }

    /// <summary>
    /// Stops the SFX source
    /// </summary>
    /// <param name="source">The source to stop</param>
    public void StopSFX(AudioSource source)
    {
        foreach(AudioSource sfxSource in _SFXSources.ToList())
        {
            if(sfxSource.clip == source.clip)
            {
                //Stop the source and remove it from the list
                source.Stop();
                _SFXSources.Remove(source);
                break;
            }
        }
    }

    /// <summary>
    /// Sets the volume of the music source and can be optionally set to overwrite the music volume setting
    /// </summary>
    /// <param name="volume">The new volume value to set</param>
    /// <param name="overwrite">Whether or not the existing volume setting should be overwritten</param>
    public void SetMusicVolume(float volume, bool overwrite = false)
    {
        _MusicSource.volume = volume;   //Set the volume

        if (overwrite)
        {
            //Update the setting and write to the XML
            SettingsManager.Instance.Settings[Constants.Settings.MusicVolume] = volume;
            SettingsManager.Instance.WriteSettings();
        }
    }

    /// <summary>
    /// Sets the volume of the SFX sources and can be optionally set to overwrite the SFX volume setting
    /// </summary>
    /// <param name="volume">The new volume value to set</param>
    /// <param name="overwrite">Whether or not the existing volume setting should be overwritten</param>
    public void SetSFXVolume(float volume, bool overwrite = false)
    {
        foreach (AudioSource sfxSource in _SFXSources)
        {
            //Loop through and set the volume for all SFX sources
            sfxSource.volume = volume;
        }

        if (overwrite)
        {
            //Update the setting and write to the XML
            SettingsManager.Instance.Settings[Constants.Settings.SFXVolume] = volume;
            SettingsManager.Instance.WriteSettings();
        }
    }

    /// <summary>
    /// Returns the volume of the music source (NOT the setting)
    /// </summary>
    /// <returns></returns>
    public float GetMusicVolume()
    {
        return _MusicSource.volume;
    }

    /// <summary>
    /// Returns the SFX volume setting
    /// </summary>
    /// <returns></returns>
    public float GetSFXVolume()
    {
        return Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.SFXVolume]);
    }

    /// <summary>
    /// Plays an AudioFile
    /// </summary>
    /// <param name="file">The AudioFile to play</param>
    /// <param name="shouldLoop">Should it loop?</param>
    /// <param name="volume">How loud should it be?</param>
    /// <returns>The audio source for the file</returns>
    public AudioSource PlayFile(AudioFile file, bool shouldLoop = false, float volume = -1.0f)
    {
        if (file != null)
        {
            if (file.Type == Constants.AudioType.SFX)
            {
                return PlaySFX(file.Clip, shouldLoop, volume);
            }

            else if (file.Type == Constants.AudioType.Music)
            {
                PlayMusic(file.Clip, shouldLoop);
            }
        }

        return null;
    }

    /// <summary>
    /// Stops an AudioFile
    /// </summary>
    /// <param name="file">The file to stop</param>
    public void StopFile(AudioFile file)
    {
        if (file != null)
        {
            if(file.Type == Constants.AudioType.SFX)
            {
                //Let's find it
                foreach(AudioSource source in _SFXSources.ToList())
                {
                    if(source.clip == file.Clip)
                    {
                        //Found it, stop it
                        StopSFX(source);
                        break;
                    }
                }
            }

            else if (file.Type == Constants.AudioType.Music)
            {
                //It's music so we can just stop it immediately
                StopMusic();
            }
        }
    }

    /// <summary>
    /// Loads an AudioFile from the specified XML Element data
    /// </summary>
    /// <param name="elem">The XML element for the AudioFile</param>
    /// <returns>The created AudioFile</returns>
    private AudioFile LoadFile(XElement elem)
    {
        AudioClip clip = (AudioClip)Resources.Load(elem.Value);
        Constants.AudioType type = (Constants.AudioType)Enum.Parse(typeof(Constants.AudioType), elem.Attribute("Type").Value);
        return new AudioFile(clip, type);
    }

    /// <summary>
    /// Parses all AudioFiles from the XML file
    /// </summary>
    /// <returns>Was the parse successful?</returns>
    private bool ParseAudioFiles()
    {
        try
        {
            //Load our XML
            TextAsset xmlText = (TextAsset)Resources.Load("AudioFiles");
            XDocument audioXML = XDocument.Parse(xmlText.text);

            XElement uiAudioElem = audioXML.Root.Element("UI");

            if(uiAudioElem != null)
            {
                //Load all of our UI SFX files
                LoadedFiles[Constants.UIButtonSelectAudioID] = LoadFile(uiAudioElem.Element("ButtonSelect"));
                LoadedFiles[Constants.UIButtonBackAudioID] = LoadFile(uiAudioElem.Element("ButtonBack"));
                LoadedFiles[Constants.UIButtonFocusAudioID] = LoadFile(uiAudioElem.Element("ButtonFocus"));
                LoadedFiles[Constants.UIButtonInactiveSelectAudioID] = LoadFile(uiAudioElem.Element("ButtonInactiveSelect"));
                LoadedFiles[Constants.UIButtonInactiveFocusAudioID] = LoadFile(uiAudioElem.Element("ButtonInactiveFocus"));
                LoadedFiles[Constants.UIPageTransToAudioID] = LoadFile(uiAudioElem.Element("PageTransTo"));
                LoadedFiles[Constants.UIPageTransFromAudioID] = LoadFile(uiAudioElem.Element("PageTransFrom"));
                LoadedFiles[Constants.UIPageTransBackToAudioID] = LoadFile(uiAudioElem.Element("PageTransBackTo"));
                LoadedFiles[Constants.UIPageTransBackFromAudioID] = LoadFile(uiAudioElem.Element("PageTransBackFrom"));

                //Now loop through and load all our AudioFiles
                foreach (XElement audioFileElem in audioXML.Root.Element("Misc").Elements("AudioFile"))
                {
                    string id = audioFileElem.Attribute("ID").Value;
                    LoadedFiles[id] = LoadFile(audioFileElem);
                }

                //Finally, let's load all our AudioGroups
                foreach(XElement audioGroupElem in audioXML.Root.Elements("AudioGroup"))
                {
                    string groupID = audioGroupElem.Attribute("ID").Value.ToString();
                    Dictionary<string, AudioFile> files = new Dictionary<string, AudioFile>();

                    foreach(XElement audioFileElem in audioGroupElem.Descendants("AudioFile"))
                    {
                        string fileID = audioFileElem.Attribute("ID").Value.ToString();
                        files[fileID] = LoadFile(audioFileElem);
                    }

                    AudioGroups[groupID] = new AudioGroup(groupID, files);
                }

                return true;    //Loading was successful!
            }

            else
            {
                Debug.LogError("ERROR: The UI audio element was null, the XML is malformed.");
            }

            return false;   //Something went wrong
        }

        catch(Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when parsing audio files. The exception is: " + ex);
            return false;
        }
    }

    private void Update()
    {
        if(!_MusicSource.isPlaying && !_DetectedMusicStopped)
        {
            //Raise the event
            OnMusicFinished(new MusicFinishedEventArgs(_MusicSource.clip));
            _DetectedMusicStopped = true;
        }
    }
}
